function x0 = get_x0(CDX, dates, method5, method2)
% --------------------------------------------------------------------------------------------------
% Return a vector of time series with parameters [k, theta_total, sigma, L_total, mu, omega1, omega2] and the
% liquidity premium of (i) CDS vs. index and (ii) tranches vs. index.
% --------------------------------------------------------------------------------------------------
% CDX                           ... credit index structure (see 'all_steps_in_a_row.m')
% dates                         ... dates for which parameters are desired, default: all
% method5                       ... which definition to use for the 5th parameters
%                                   'mu' = expected jump size \mu (default)
%                                   'mu*L' = jump intensity x jump size
% method2                       ... which definition to use for the 2nd parameter
%                               ... 'thet' = theta_total
%                               ... 'thet*k' = theta_total * k
% --------------------------------------------------------------------------------------------------
% sample call: get_x0(CDX_NA_IG2_monthly, CDX_NA_IG2_monthly.dates{1})
% --------------------------------------------------------------------------------------------------

if (nargin < 2)
    dates = CDX.dates{1};
end
if (nargin < 3)
    method5 = 'mu*L';
end
if (nargin < 4)
    method2 = 'thet*k';
end

% Use first CDS in portfolio (AA) to extract parameters => not allowed to have missing data
cds = CDX.portfolio(1);
[trash, used_pos_cdx, used_pos_cds, used_pos_params] = intersect_triple(CDX.dates{1}, cds.dates{1}, dates);
if (length(used_pos_cds) < length(dates))
    error('Desired dates are not subset of CDS dates');
end
AJD = CDX.AJD_common_factor;

% Extract relevant ai series
if (length(cds.ai) > 1)
    ai = cds.ai(used_pos_cds);
else
    ai = cds.ai;
end

% k
x0 = {};
if (length(AJD.k) == 1)
    x0{1} = AJD.k;
else
    x0{1} = AJD.k(used_pos_cdx);
end

% theta_total (theta_total*k for other method)
if (length(cds.AJD.theta) == 1)
    if strcmp(method2, 'thet')
        x0{2} = cds.AJD.theta(used_pos_cds) ./ ai + AJD.theta(used_pos_cdx);
    elseif strcmp(method2, 'thet*k')
        x0{2} = (cds.AJD.theta(used_pos_cds) ./ ai + AJD.theta(used_pos_cdx)) * AJD.k;
    end
else
    if strcmp(method2, 'thet')
        x0{2} = cds.AJD.theta(used_pos_cds) ./ ai + AJD.theta(used_pos_cdx);
    elseif strcmp(method2, 'thet*k')
        x0{2} = (cds.AJD.theta(used_pos_cds) ./ ai + AJD.theta(used_pos_cdx)) .* AJD.k(used_pos_cdx);
    end
end

% sigma
if (length(AJD.sigma) == 1)
    x0{3} = AJD.sigma;
else
    x0{3} = AJD.sigma(used_pos_cdx);
end

% L_total
if (length(cds.AJD.L) == 1)
    x0{4} = AJD.L + cds.AJD.L;
else
    x0{4} = AJD.L(used_pos_cdx) + cds.AJD.L(used_pos_cds);
end

% mu (L_total * mu for other method)
if (length(AJD.mu) == 1)
    if strcmp(method5, 'mu')
        x0{5} = AJD.mu;
    elseif strcmp(method5, 'mu*L')
        x0{5} = AJD.mu * x0{4};
    end
else
    if strcmp(method5, 'mu')
        x0{5} = AJD.mu(used_pos_cdx);
    elseif strcmp(method5, 'mu*L')
        x0{5} = AJD.mu(used_pos_cdx) .* x0{4};
    end
end

% omega1
if (length(cds.AJD.L) == 1)
    x0{6} = AJD.L / (AJD.L + cds.AJD.L);
else
    x0{6} = AJD.L(used_pos_cdx) ./ (AJD.L(used_pos_cdx) + cds.AJD.L(used_pos_cds));
end

% omega2
if (length(AJD.theta) == 1)
    x0{7} = cds.ai * AJD.theta / (cds.ai * AJD.theta + cds.AJD.theta);   %AJD.theta * ai / cds.AJD.theta;
else
    x0{7} = cds.ai(used_pos_cds) .* AJD.theta(used_pos_cdx) ./ (cds.ai(used_pos_cds) .* AJD.theta(used_pos_cdx) + cds.AJD.theta(used_pos_cds)); 
end

% liq_prem of CDS vs. index
if (length(CDX.liq_prem_cds) == 1)
    x0{8} = CDX.liq_prem_cds;
else
    x0{8} = CDX.liq_prem_cds(used_pos_cdx);
end

% liq_prem of tranches vs. index
if (length(CDX.liq_prem_tranches) == 1)
    x0{9} = CDX.liq_prem_tranches;
else
    x0{9} = CDX.liq_prem_tranches(used_pos_cdx);
end
